# 機能設計書 6-制御フロー（Control Flow Operations）

## 概要

本ドキュメントは、TensorFlowのコアフレームワークにおける制御フロー（Control Flow Operations）機能の設計を記述する。条件分岐（cond）・ループ（while_loop）・switch/case・assertなどのグラフ制御フローを提供する。

### 本機能の処理概要

制御フロー機能は、TensorFlowの計算グラフ内で条件分岐やループなどの制御構造を実現する操作群を提供する。グラフモードでの動的な制御フロー（条件分岐やwhile_loop）とEagerモードでのPythonネイティブな制御フローの両方をサポートし、AutoGraphによるPythonコードからの自動変換も含む。

**業務上の目的・背景**：機械学習モデルの多くは、条件分岐（例：学習時と推論時で異なる処理）やループ（例：RNNのタイムステップ展開、動的反復計算）を含む。グラフ実行モードではPythonの制御構文が直接使えないため、TensorFlowの制御フロー操作が必要となる。本機能はこれらの制御構造をグラフ内で表現可能にする。

**機能の利用シーン**：(1) tf.condによる学習/推論の条件分岐、(2) tf.while_loopによるRNNの動的展開、(3) Switch/Mergeによる低レベル制御フロー、(4) Assertによるランタイム検証、(5) AutoGraphによるPython制御構文の自動変換。

**主要な処理内容**：
1. Switch/RefSwitch: 述語に基づくデータ経路切り替え
2. Merge/RefMerge: 複数入力からの最初の利用可能値の選択
3. Enter/RefEnter: ループフレームへの進入
4. Exit/RefExit: ループフレームからの退出
5. NextIteration/RefNextIteration: ループ次回反復へのテンソル受け渡し
6. LoopCond: while_loopの終了条件評価
7. _SwitchN: N分岐のスイッチ操作
8. Abort: グラフ実行の中断

**関連システム・外部連携**：AutoGraphモジュール（tensorflow/python/autograph/）と連携し、Pythonのif/for/while文を自動的に対応するTensorFlow制御フロー操作に変換する。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はコアAPIのため直接的な画面関連はなし |

## 機能種別

計算処理（グラフ制御フロー処理）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| data | tf.Tensor | Yes | 制御対象のデータテンソル | 任意の型 |
| pred | tf.Tensor(bool) | Switch時 | 分岐条件（スカラーbool） | ランク0であること |
| frame_name | string | Enter時 | ループフレーム名 | - |
| is_constant | bool | Enter時 | ループ内定数フラグ | - |

### 入力データソース

Python API呼び出し時の引数、またはAutoGraphによる自動生成。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output_false / output_true | tf.Tensor | Switch時の各分岐出力 |
| output | tf.Tensor | Merge/Enter/Exit/NextIteration時の出力 |
| value_index | tf.Tensor(int32) | Merge時の選択インデックス |

### 出力先

計算グラフ内の後続ノードに接続される。

## 処理フロー

### 処理シーケンス

```
1. Python API呼び出し（tf.cond, tf.while_loop等）
   └─ control_flow_ops.py内の高レベルAPI
2. 低レベル制御フロー操作への分解
   └─ Switch/Merge/Enter/Exit/NextIteration/LoopCond
3. 入力テンソルの変換・検証
   └─ _Identity, _NextIteration, _Enter 内部関数
4. gen_control_flow_ops経由でC++カーネル呼び出し
5. ランタイムでの制御フロー実行
```

### フローチャート

```mermaid
flowchart TD
    A[tf.cond/tf.while_loop] --> B{Eager実行?}
    B -->|Yes| C[Pythonネイティブ制御フロー]
    B -->|No| D[グラフ制御フロー構築]
    D --> E[Switch/Merge/Enter/Exit生成]
    E --> F[計算グラフに制御フローノード追加]
    C --> G[結果返却]
    F --> G
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 述語スカラー制約 | Switchの述語（pred）はスカラーboolでなければならない | Switch操作時 |
| BR-02 | フレーム名一意性 | ループフレーム名はグラフ内で一意でなければならない | Enter操作時 |
| BR-03 | Ref型サポート | 変数参照型テンソルに対してはRef系操作を使用 | Ref型テンソル時 |
| BR-04 | CompositeTensorサポート | CompositeTensorに対してはnest.map_structureで再帰適用 | CompositeTensor時 |

### 計算ロジック

- Switch: pred=True → output_true、pred=False → output_false
- Merge: N個の入力のうち最初に利用可能になったテンソルを出力
- while_loop: LoopCondがFalseを返すまでbodyを反復実行

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作は発生しない |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgumentError | ランクエラー | Switch述語が非スカラー | スカラーboolテンソルを使用する |
| TypeError | 型エラー | 非テンソル・非CompositeTensor入力 | 有効なテンソルを使用する |
| AbortedError | 中断 | Abort op実行時 | 意図的な中断、エラーメッセージを確認 |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- while_loopの各反復で不要なメモリコピーを避けること
- 制御フロー操作自体のオーバーヘッドを最小化すること

## セキュリティ考慮事項

- while_loopの無限ループ防止（maximum_iterations引数）
- Abort操作によるサービス停止への注意

## 備考

- Eagrモードではほとんどの制御フロー操作はPythonのネイティブ制御構文で代替される
- AutoGraphはtf.functionデコレータ内でPythonのif/for/whileを自動変換する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

制御フロー操作が扱うテンソル型（通常テンソル、Ref型テンソル、CompositeTensor）を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | control_flow_ops.py | `tensorflow/python/ops/control_flow_ops.py` | _Identity, _NextIteration, _Enter内部関数 |

**読解のコツ**: 制御フロー操作は通常テンソルとRef型テンソルで異なるopを使用する。_is_ref_dtypeプロパティで判定し、Ref型にはref_系関数を使用する（例: ref_identity, ref_next_iteration）。

#### Step 2: エントリーポイントを理解する

Python側の内部関数とC++ op定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | control_flow_ops.py | `tensorflow/python/ops/control_flow_ops.py` | 内部関数群 |

**主要処理フロー**:
1. **59-82行目**: `_Identity` 関数 - テンソルのアイデンティティコピー（型判定あり）
2. **85-96行目**: `_NextIteration` 関数 - ループ次回反復へのテンソル受け渡し
3. **99行目以降**: `_Enter` 関数 - ループフレームへの進入

#### Step 3: C++オペレーション定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | control_flow_ops.cc | `tensorflow/core/ops/control_flow_ops.cc` | REGISTER_OPによる制御フローop定義 |

**主要処理フロー**:
- **30-44行目**: `SwitchShape` 関数 - Switch opの形状推論
- **46-64行目**: `SwitchNShape` 関数 - _SwitchN opの形状推論
- **68-75行目**: `Switch` op登録 - 述語による二分岐
- **77-84行目**: `RefSwitch` op登録 - Ref型のSwitch
- **86-92行目**: `_SwitchN` op登録 - N分岐スイッチ
- **95-100行目**: `RefSelect` op登録 - Ref型の選択

### プログラム呼び出し階層図

```
tf.cond(pred, true_fn, false_fn)
    |
    +-- control_flow_ops.py
    |       |
    |       +-- _Identity(tensor)
    |       |       +-- gen_array_ops.identity() [通常テンソル]
    |       |       +-- gen_array_ops.ref_identity() [Ref型]
    |       |
    |       +-- Switch(data, pred)
    |               +-- gen_control_flow_ops.switch()
    |
    +-- C++ Switch kernel
            +-- SwitchShape (形状推論)
```

### データフロー図

```
[入力]                    [処理]                        [出力]

data, pred         -->  control_flow_ops.py        -->  output_true/false
(テンソル, bool)        (型判定・分岐構築)
                            |
                    control_flow_ops.cc
                    (SwitchShape推論)
                            |
                    C++ Kernel
                    (Switch/Merge/Enter/Exit)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| control_flow_ops.py | `tensorflow/python/ops/control_flow_ops.py` | ソース | Python制御フローAPI |
| control_flow_ops.cc | `tensorflow/core/ops/control_flow_ops.cc` | ソース | C++ op定義 |
| gen_control_flow_ops.py | `tensorflow/python/ops/gen_control_flow_ops.py` | 自動生成 | Python バインディング |
| control_flow_util.py | `tensorflow/python/ops/control_flow_util.py` | ソース | 制御フローユーティリティ |
| tensor_array_ops.py | `tensorflow/python/ops/tensor_array_ops.py` | ソース | TensorArray操作（while_loop内で使用） |
| control_flow_pb2.py | `tensorflow/core/protobuf/control_flow_pb2.py` | 自動生成 | Protocol Buffer定義 |
